struct PxBakedNames
(
	PxBakedName = "bake",
	PxBakedNameMassFXPos = "MassFX Baked Position",
	PxBakedNameMassFXRot = "MassFX Baked Rotation"
)
global PxBakedNames = PxBakedNames()

fn PxNeedToBake modRB =
(
	(modRB.baked == 0) and ((modRB.type == PX_PHYSTYPE_DYNAMIC) or (modRB.type == PX_PHYSTYPE_KINEMATIC and modRB.switchType))
)

fn pxBakeClothing =
(
	progressStart  nvpxText.bakeProgressCaptionClothing
	PxStopSimulation()
	local setKeyWasOn = maxOps.getSetKeyMode()
	if setKeyWasOn then
		maxOps.setKeyMode  = false
	slidertime = animationRange.start
	nvpx.ClearInvalidNodes()
	nvpx.SetBehaviorReason gPxReasonBake gPxReasonModifierNone
	
	PxCreatePhysXScene()

	delta = 1.0/frameRate

	local timeCounter = 0
	local animationLen = animationRange.end - animationRange.start

	nvpx.SetAnimationState(true)
	
	while (timeCounter <= animationLen) do
	(	
		timeCounter = timeCounter + 1	
		
		if not progressupdate (timeCounter * 100 / animationLen + 1) then
		(
			PxStopSimulation()
			slidertime = animationRange.start
			nvpx.SetBehaviorReason gPxReasonPlayback gPxReasonModifierCancel
			progressend()
			if setKeyWasOn then
				maxOps.setKeyMode  = true
			return()
		)
		
		with redraw off
		(
			undo off
			(
				try
				(
					PxSimulateOneFrame()
				)catch ()
			)
		)
		redrawviews()
	)
	undo off
	(
		PxStopSimulation()
		if setKeyWasOn then
			maxOps.setKeyMode  = true
		slidertime = animationRange.start
	)
	nvpx.SetBehaviorReason gPxReasonPlayback gPxReasonModifierNone
	progressend()
)

fn checkBakedPositionName name =
(
   (name == PxBakedNames.PxBakedName) or (name == PxBakedNames.PxBakedNameMassFXPos)
)

fn checkBakedRotationName name =
(
   (name == PxBakedNames.PxBakedName) or (name == PxBakedNames.PxBakedNameMassFXRot)
)

-- Sets up List Controllers with weighting for each channel (old and new) switching on or off,
-- according to when they apply.  New keys are absolute, and overwrite exiting animation.
fn bakeSetupController_AbsoluteMode node  =
(
	local modRB = PxGetModRB node

	if modRB != undefined and ((classof node.controller) == prs) do
	(
		-- Always apply new list controllers. They will be removed during unbake.
		-- If the artist applies Rigid Bodifier, bakes, removes Rigid Body modifier, and repeats,
		-- this creates two nested list controllers, by design; the first controller is not reused or clobbered

		local posCtrl = Position_List()  -- Position List Controller
		posCtrl.available.controller = Position_XYZ()  -- Sets bake keys as 1
		node.position.controller = posCtrl  -- Sets previous controller as slot 2 (not obvious)
		posCtrl.weight[1].controller = (Boolean_Float())
		posCtrl.weight[2].controller = (Boolean_Float())
		posCtrl.setName 1 nvpxText.bakeControllerPosName
		posCtrl.setActive 1  -- Bake keys are the active, keyframable slot


		local rotCtrl = Rotation_List()  -- Rotation List Controller
		rotCtrl.available.controller = Euler_XYZ()  -- Sets bake keys as slot 1
		node.rotation.controller = rotCtrl  -- Sets previous controller as slot 2 (not obvious)
		rotCtrl.weight[1].controller = (Boolean_Float())
		rotCtrl.weight[2].controller = (Boolean_Float())
		rotCtrl.setName 1 nvpxText.bakeControllerRotName
		rotCtrl.setActive 1  -- Bake keys are the active, keyframable slot
		

		-- Set an animated switch frame.  Needed to support Kinematic to Dynamic switching

		local switchTime =
			if ((modRB.type==PX_PHYSTYPE_KINEMATIC) and modRB.switchType) then 
				modRB.switchTypeAtFrame -- Switch weighting to the baked controller at the appropriate time
			else
				animationRange.start -- Full weighting on the baked controller at all times

		with animate off  -- initial weighting, use original controller, don't use baked controller
		(
			posCtrl.weight[1] = rotCtrl.weight[1] = 0.0
			posCtrl.weight[2] = rotCtrl.weight[2] = 100.0
		)
		
		-- first, turn off the autokey default key so we can be sure that there will be no
		-- interference to the keys we want to create from the key generated by auto-key
		local defaultKeyOn = maxOps.autoKeyDefaultKeyOn
		maxOps.autoKeyDefaultKeyOn = false

		with animate on  ---- animated switched weighting, use baked controller, don't use original controller
		(
			-- key the weight of the baked controller to 0 at switch time
			-- then key it to 100 at switch time + 1 tick
			at time switchTime
			(
				posCtrl.weight[1] = rotCtrl.weight[1] = 0.0
				posCtrl.weight[2] = rotCtrl.weight[2] = 100.0
			)
			at time ((switchTime as time) + 1t)
			(
				posCtrl.weight[1] = rotCtrl.weight[1] = 100.0
				posCtrl.weight[2] = rotCtrl.weight[2] = 0.0
			)

			-- set bake controller values equal to original values, at switch time
			at time (switchTime)
			(
					posCtrl[1].controller.value = posCtrl[2].controller.value
					rotCtrl[1].controller.value = rotCtrl[2].controller.value
			)
		)

		maxOps.autoKeyDefaultKeyOn = defaultKeyOn
	)
)

-- Sets up List Controllers with both channels (old and new) at full weighting.
-- New keys have an additive effect, relative to the exiting animation.
fn bakeSetupController_AdditiveMode node  =
(
	-- transform into list controller if needed
	if classof(node.position.controller) != Position_List then
		node.position.controller = position_list()
	
	if classof(node.rotation.controller) != Rotation_List then
	(
		node.rotation.controller = rotation_list()
	)
		
	--setup bake track for the position
	index = -1
	for i = 1 to node.position.controller.getCount() do
	(
		if checkBakedPositionName(node.position.controller.getName(i)) then
			index = i
	)
	
	if index == -1 then
	(
		node.position.available.controller = Position_XYZ()
		index = node.position.controller.getCount()
		-- if PxBakedNames.PxBakedName this to "MassFX Position Baked", it fails to bake some file. See MSFX-167
		node.position.controller.setName index PxBakedNames.PxBakedName
	)
	
	node.position.controller.setActive index
	
	-- setup bake track for rotation
	index = -1
	for i = 1 to node.rotation.controller.getCount() do
	(
		if checkBakedRotationName(node.rotation.controller.getName(i)) then
			index = i
	)
	
	if index == -1 then
	(
		node.rotation.available.controller = Euler_XYZ()
		index = node.rotation.controller.getCount()
		-- if rename PxBakedNames.PxBakedName to "MassFX Rotation Baked", it fails to bake some file. See MSFX-167
		node.rotation.controller.setName index PxBakedNames.PxBakedName
	)
	
	node.rotation.controller.setActive index
)

fn BakeSetupController node =
(
	if PHYSX_AUTODESK_VER!=undefined then
		BakeSetupController_AbsoluteMode node	-- Autodesk style
	else
		BakeSetupController_AdditiveMode node	-- NVidia style
)



-- Removes baked keyframes from List Controllers,
-- and reverts the controller structure to an identical state as before baking.
fn bakeUnsetupController_Revert node =
(
	if ((classof node.controller) == prs) do
	(
		local posCtrl = node.position.controller
		if ((classof posCtrl) ==Position_List) do
		(
			local foundIndex = undefined
			for i = 1 to posCtrl.count while foundIndex==undefined do
				if (posCtrl.getName i) == nvpxText.bakeControllerPosName do
					foundIndex=i
			
			if foundIndex!=undefined do  -- List controller appears to have the expected format, proceed
			(	-- Delete the baked key controller
				posCtrl.delete foundIndex
				-- If only one subcontroller is left (the usual case), replace the list controller with it				
				if posCtrl.count == 1 do
					node.position.controller = posCtrl[1].controller
			)
				
		)

		local rotCtrl = node.rotation.controller
		if ((classof rotCtrl) ==Rotation_List) do
		(
			local foundIndex = undefined
			for i = 1 to rotCtrl.count while foundIndex==undefined do
				if (rotCtrl.getName i) == nvpxText.bakeControllerRotName do
					foundIndex=i
			
			if foundIndex!=undefined do  -- List controller appears to have the expected format, proceed
			(	-- Delete the baked key controller
				rotCtrl.delete foundIndex
				-- If only one subcontroller is left (the usual case), replace the list controller with it
				if rotCtrl.count == 1 do
					node.rotation.controller = rotCtrl[1].controller
			)
		)
	)
)

-- Removes baked keyframes from List Controllers,
-- but does not revert the controller structure to its state before baking.
fn bakeUnsetupController_DontRevert node =
(
	if ((classof node.controller) == prs) do
	(
		-- reset controller to a non bake one

		local posCtrl = node.position.controller
		if ((classof posCtrl) ==Position_List) do
		(
			deletekeys posCtrl[posCtrl.active] #allKeys

			index = -1
			for i = 1 to posCtrl.getCount() do
			(
				if not checkBakedPositionName(posCtrl.getName(i)) then
					index = i
			)
	
			posCtrl.setActive index
		)

		local rotCtrl = node.rotation.controller
		if ((classof rotCtrl) ==Rotation_List) do
		(
			deletekeys rotCtrl[rotCtrl.active] #allkeys

			index = -1
			for i = 1 to rotCtrl.getCount() do
			(
				if not checkBakedRotationName(rotCtrl.getName(i)) then
					index = i
			)

			rotCtrl.setActive index	
		)

		-- TO DO: Is this really a good idea?
		-- And shouldn't it happen before baking, not after?
		deletekeys node.scale.controller #allkeys
	)
)

fn BakeUnsetupController node =
(
	if PHYSX_AUTODESK_VER!=undefined then
		BakeUnsetupController_Revert node	-- Autodesk style
	else
	(
		BakeUnsetupController_DontRevert node	-- NVidia style
	)
)

fn pxRestoreBake nodes containsBiped = 
(
	PxStopSimulation()
	slidertime = animationRange.start
	nvpx.SetBehaviorReason gPxReasonPlayback gPxReasonModifierCancel	
	progressend()

	if containsBiped then
		nvpx.BipedSetKeys false
			
	for node in nodes do
	(
		if (classof node.baseobject  == biped_object) then
			biped.deleteKeys node.controller #allKeys 
				
		if  (classof node.controller == Vertical_Horizontal_Turn) then
		(
			biped.deletekeys  node.controller.vertical #allkeys   
			biped.deletekeys  node.controller.horizontal #allkeys
			biped.deletekeys  node.controller.turning #allkeys
		)
		-- make sure we erase the bake controllers
		-- setup list controller only if not biped
		if (classof node.baseobject  != biped_object) then
			bakeUnsetupController node
	)
	-- Garbage collect, deletes references to controllers lingering in memory
	-- Otherwise the UI may incorrectly indicate some remaining controllers are instances
	gc()
)

fn pxRemoveInstances nodes =
(
	bad = #()
	good = #()
	
	for node in nodes do
	(
		if findItem bad node != 0 then
			continue()
		
		append good node
		
		if InstanceMgr.GetInstances node &instances > 1 then
			for temp in instances do
				if temp != node then
					append bad temp
	)
	
	return good;
)

fn pxBake nodes containsBiped =
(
	progressStart nvpxText.bakeProgressCaptionRegular
	PxStopSimulation()
	local setKeyWasOn = maxOps.getSetKeyMode()
	if setKeyWasOn then
		maxOps.setKeyMode  = false
	slidertime = animationRange.start

	delta = 1.0/frameRate

	local timeCounter = 0
	local animationLen = animationRange.end - animationRange.start
	nvpx.SetAnimationState(true)
	
	if containsBiped then
		nvpx.BipedSetKeys true
	
	for node in nodes do
	(
		-- setup list controller only if not biped
		if (classof node.baseobject  != biped_object) and (classof node.controller != Vertical_Horizontal_Turn) then
			bakeSetupController node
	)
	
	PxCreatePhysXScene()
	
	while (timeCounter <= animationLen) do
	(
		timeCounter = timeCounter + 1
		
		if not progressupdate (timeCounter * 100 / animationLen + 1) then
		(
			pxRestoreBake nodes containsBiped
			if setKeyWasOn then
				maxOps.setKeyMode  = true
			return()
		)
		
		with redraw off
		(
			PxSimulateOneFrame();
		)
		redrawviews ();
	)
		
	-- no need to set to kinematic for all instances, one is enough
	nodes = pxRemoveInstances nodes
	
	for node in nodes do
	(
			modRB = PxGetModRB node
			
			if modRB != undefined then
			(
				if modRB.type == PX_PHYSTYPE_DYNAMIC do modRB.baked = PX_PHYSBAKED_DYNAMIC
				if modRB.type == PX_PHYSTYPE_KINEMATIC do modRB.baked = PX_PHYSBAKED_KINEMATIC
				modRB.type = PX_PHYSTYPE_KINEMATIC
			)
	)
	
	nvpx.SetBehaviorReason gPxReasonPlayback gPxReasonModifierNone

	
	undo off
	(
		PxStopSimulation()
		if setKeyWasOn then
			maxOps.setKeyMode  = true
		slidertime = animationRange.start

		if containsBiped then
			nvpx.BipedSetKeys false		
	)
	

	progressend()
)

fn pxUnbake nodes =
(
	local previousTime = sliderTime
	sliderTime = animationrange.start
	
	
	noInstances = pxRemoveInstances nodes

	for node in nodes do
	(
		if  (classof node.controller == Vertical_Horizontal_Turn) then
		(
			fromBaking = nvpx.GetBakedFlag(node) --getUserProp node "PhysxRagdollBaking"
			
			if fromBaking != undefined and fromBaking == true then
			(
				biped.deletekeys  node.controller.vertical #allkeys   
				biped.deletekeys  node.controller.horizontal #allkeys
				biped.deletekeys  node.controller.turning #allkeys
				
				nvpx.SetBakedFlag node false
			)
		)
		else
		(
			-- no need to set to previous mode for all instances, one is enough
			if findItem noInstances node != 0 then
			(
				modRB = PxGetModRB node
				if modRB.baked != 0 then
				(
					if modRB.baked==PX_PHYSBAKED_KINEMATIC do modRB.type = PX_PHYSTYPE_KINEMATIC
					if modRB.baked==PX_PHYSBAKED_DYNAMIC do modRB.type = PX_PHYSTYPE_DYNAMIC

					-- following 2 lines are for compatibility with old files.
					if modRB.baked==5 do modRB.type = PX_PHYSTYPE_DYNAMIC
					if modRB.baked==6 do modRB.type = PX_PHYSTYPE_KINEMATIC
				)
				modRB.baked = 0
			)
		
			if (classof node.baseobject  == biped_object) then
				biped.deleteKeys node.controller #allKeys 
			else
			(
				bakeUnsetupController node			-- set reset active controller
			)
		)
	)
	
	-- Garbage collect, deletes references to controllers lingering in memory
	-- Otherwise the UI may incorrectly indicate some remaining controllers are instances
	gc()
	sliderTime = previousTime
)

fn pxBakeAll remove =
(
	nodes = #()
	containsBiped = false
	
	if remove then 
		nvpx.SetBehaviorReason gPxReasonUnbake gPxReasonModifierNone
	else
		nvpx.SetBehaviorReason gPxReasonBake gPxReasonModifierNone

	for node in geometry do
	(
		if classof node.baseobject == biped_object then
			containsBiped = true

		if classof node.controller == Vertical_Horizontal_Turn then
			append nodes node
			
		local modRB = PxGetModRB node					
		
		if (modRB != undefined) then
		(
			if remove then
			(	
				if (modRB.baked != 0) then 
					append nodes node
			)
			else if PxNeedToBake(modRB) then
				append nodes node
		)
	)
	
	--if nodes.count > 0 then
	(
		if remove then
			pxUnbake nodes
		else
			pxBake nodes containsBiped
	)

	nvpx.SetBehaviorReason gPxReasonPlayback gPxReasonModifierNone
	
	px_selectionChanged()  -- Update PhysX Tools dialog
)

fn pxBakeNodes nodes remove =
(
	local finalNodeList = #()
	local containsBiped = false
	
	if remove then 
		nvpx.SetBehaviorReason gPxReasonUnbake gPxReasonModifierSelected
	else
		nvpx.SetBehaviorReason gPxReasonBake gPxReasonModifierSelected
	
	for node in nodes do
	(
		if classof node.baseobject == biped_object then
			containsBiped = true
		else
		(
			local modRB = PxGetModRB node
			
			if (modRB != undefined) then
			(
				if remove then
				(
					if (modRB.baked != 0) then 
						append finalNodeList node
				)
				else if PxNeedToBake(modRB) then
					append finalNodeList node
			)
		)
	)

	if containsBiped then
	(
		-- need to add the whole biped hiearchy to the node list
		for obj in geometry do
		(				
			if classof obj.baseobject == biped_object then 
			(
				local modRB = PxGetModRB obj
				if (modRB != undefined) and PxNeedToBake(modRB) then
					append finalNodeList obj
				
				if modRB != undefined and remove and (modRB.baked != 0) then
					append finalNodeList obj
				
				if classof obj.controller == Vertical_Horizontal_Turn then
					append finalNodeList obj
			)
		)
	)	
	
	--if finalNodeList.count > 0 then
	(
		if remove then
			pxUnbake finalNodeList
		else
			pxBake finalNodeList containsBiped
	)
	
	nvpx.SetBehaviorReason gPxReasonPlayback gPxReasonModifierNone
	
	
	px_selectionChanged()  -- Update PhysX Tools dialog
)

fn pxBakeSelection remove =
(
	pxBakeNodes $selection remove
)

-- If the signature of this function changes, please update pluginInterfaces::BakeNodes accordingly
fn pxBakeNodesUndoOn nodes remove =
(
	local undoName
	if (remove == true) then
	(
		undoName = nvpxText.toolsDlgBakeSelected
	)
	else
	(
		undoName = nvpxText.toolsDlgUnbakeSelected
	)
	undo undoName on
	(
		pxBakeNodes nodes remove
	)
)

fn pxBakeSelectionUndoOn remove =
(
	pxBakeNodesUndoOn $selection remove
)
-------BEGIN-SIGNATURE-----
-- 4wYAADCCBt8GCSqGSIb3DQEHAqCCBtAwggbMAgEBMQ8wDQYJKoZIhvcNAQELBQAw
-- CwYJKoZIhvcNAQcBoIIE3jCCBNowggPCoAMCAQICEDUAFkMQxqI9PltZ2eUG16Ew
-- DQYJKoZIhvcNAQELBQAwgYQxCzAJBgNVBAYTAlVTMR0wGwYDVQQKExRTeW1hbnRl
-- YyBDb3Jwb3JhdGlvbjEfMB0GA1UECxMWU3ltYW50ZWMgVHJ1c3QgTmV0d29yazE1
-- MDMGA1UEAxMsU3ltYW50ZWMgQ2xhc3MgMyBTSEEyNTYgQ29kZSBTaWduaW5nIENB
-- IC0gRzIwHhcNMTkwNjI1MDAwMDAwWhcNMjAwODA3MjM1OTU5WjCBijELMAkGA1UE
-- BhMCVVMxEzARBgNVBAgMCkNhbGlmb3JuaWExEzARBgNVBAcMClNhbiBSYWZhZWwx
-- FzAVBgNVBAoMDkF1dG9kZXNrLCBJbmMuMR8wHQYDVQQLDBZEZXNpZ24gU29sdXRp
-- b25zIEdyb3VwMRcwFQYDVQQDDA5BdXRvZGVzaywgSW5jLjCCASIwDQYJKoZIhvcN
-- AQEBBQADggEPADCCAQoCggEBAMsptjSEm+HPve6+DClr+K4CgrtrONjtHxHBwTMC
-- mrwF9bnsdMiSgvYigTKk858TlqVs7GiBVLD3SaSZqfSXOv7L55i965L+wIx0EZxX
-- xDzbyLh1rLSSNWO8oTDIKnPsiwo5x7CHRUi/eAICOvLmz7Rzi+becd1j/JPNWe5t
-- vum0GL/8G4vYICrhCycizGIuv3QFqv0YPM75Pd2NP0V4W87XPeTrj+qQoRKMztJ4
-- WNDgLgT4LbMBIZyluU8iwXNyWQ8FC2ya3iJyy0EhZhAB2H7oMrAcV1VJJqwZcZQU
-- XMJTD+tuCqKqJ1ftv1f0JVW2AADnHgvaB6E6Y9yR/jnn4zECAwEAAaOCAT4wggE6
-- MAkGA1UdEwQCMAAwDgYDVR0PAQH/BAQDAgeAMBMGA1UdJQQMMAoGCCsGAQUFBwMD
-- MGEGA1UdIARaMFgwVgYGZ4EMAQQBMEwwIwYIKwYBBQUHAgEWF2h0dHBzOi8vZC5z
-- eW1jYi5jb20vY3BzMCUGCCsGAQUFBwICMBkMF2h0dHBzOi8vZC5zeW1jYi5jb20v
-- cnBhMB8GA1UdIwQYMBaAFNTABiJJ6zlL3ZPiXKG4R3YJcgNYMCsGA1UdHwQkMCIw
-- IKAeoByGGmh0dHA6Ly9yYi5zeW1jYi5jb20vcmIuY3JsMFcGCCsGAQUFBwEBBEsw
-- STAfBggrBgEFBQcwAYYTaHR0cDovL3JiLnN5bWNkLmNvbTAmBggrBgEFBQcwAoYa
-- aHR0cDovL3JiLnN5bWNiLmNvbS9yYi5jcnQwDQYJKoZIhvcNAQELBQADggEBADo7
-- 6cASiVbzkjsADk5MsC3++cj9EjWeiuq+zzKbe55p6jBNphsqLUvMw+Z9r2MpxTEs
-- c//MNUXidFsslWvWAUeOdtytNfhdyXfENX3baBPWHhW1zvbOPHQLyz8LmR1bNe9f
-- R1SLAezJaGzeuaY/Cog32Jh4qDyLSzx87tRUJI2Ro5BLA5+ELiY21SDZ7CP9ptbU
-- CDROdHY5jk/WeNh+3gLHeikJSM9/FPszQwVc9mjbVEW0PSl1cCLYEXu4T0o09ejX
-- NaQPg10POH7FequNcKw50L63feYRStDf6GlO4kNXKFHIy+LPdLaSdCQL2/oi3edV
-- MdpL4F7yw1zQBzShYMoxggHFMIIBwQIBATCBmTCBhDELMAkGA1UEBhMCVVMxHTAb
-- BgNVBAoTFFN5bWFudGVjIENvcnBvcmF0aW9uMR8wHQYDVQQLExZTeW1hbnRlYyBU
-- cnVzdCBOZXR3b3JrMTUwMwYDVQQDEyxTeW1hbnRlYyBDbGFzcyAzIFNIQTI1NiBD
-- b2RlIFNpZ25pbmcgQ0EgLSBHMgIQNQAWQxDGoj0+W1nZ5QbXoTANBgkqhkiG9w0B
-- AQsFADANBgkqhkiG9w0BAQEFAASCAQAIdQS9Rl9UnOiGhwGRT5JmNux4eaeVZy7q
-- i6b/+HiEt+1dZgFu8onrz6a3D5mYiTM2qE+4isZOpgZeFt4fkNNv3m1df+aa0U7X
-- Z2Hz3KK/4xjAMiXNOpstmtUAriDKhlwSuSuF1BhSPFxB5VOAogjTDHGcFkRskquA
-- Z/Qe8y+ZjvJZmkqlYB+VC6ONt4XMCh7DcoXxy3APwR+/gfe1HaqBeUuq5qpxNWWo
-- aNzHFJVKbuO7t1tKP7fK6kHFCc27HGDb8wkzOhZFiSDc4gKT7BTyCQ17FwPS5wP4
-- AcmAMcgWoo8fpaIRRlVUX6bhHrGPawuM2wWA2V+ZU692DDmSAj7w
-- -----END-SIGNATURE-----